#!/usr/bin/env python3

import evdev
import formatter
import os
import selectors
import subprocess
import sys

PWD = os.path.abspath(os.path.dirname(__file__))

handlers = {}

def prepare_configs ():
    confdir = os.path.join(PWD, 'conf')
    for confn in [c for c in os.listdir(confdir) if c.endswith('json')]:
        conf = formatter.readjson(os.path.join(confdir, confn))
        vmid = str(conf.get('vmid', 0))
        runner = '/tmp/{}.run'.format(vmid)
        handlers[vmid] = {'runner': runner, 'proc': None, 'fn': os.path.join(confdir, confn)}

def run_config (vm):
    conf = formatter.readjson(vm['fn'])
    frunner = open(vm['runner'], 'w')
    frunner.write(formatter.format(conf))
    frunner.close()
    vm['proc'] = subprocess.Popen(('bash', vm['runner']), stdin=subprocess.PIPE)

def handle_stdin (line):
    line = tuple(filter(None, line.strip().split(' ')))
    vm = None
    if len(line) > 0:
        if len(line) > 1:
            if line[1] in handlers:
                vm = handlers[line[1]]
        if line[0] == 'start' and vm is not None:
            if vm['proc'] is not None:
                print("VM {} already started".format(line[1]))
            else:
                print("VM {} is starting".format(line[1]))
                run_config(vm)
        elif line[0] == 'stop' and vm is not None:
            if vm['proc'] is None:
                print("VM {} not running".format(line[1]))
            else:
                vm['proc'].stdin.write(b'system_powerdown\n')
                vm['proc'].stdin.flush()
                try:
                    vm['proc'].wait(60)
                except:
                    pass
                vm['proc'] = None
                print("VM {} has stopped".format(line[1]))
        elif line[0] == 'reset' and vm is not None:
            if vm['proc'] is None:
                print("VM {} not running".format(line[1]))
            else:
                print("sendinf VM {} system_reset".format(line[1]))
                vm['proc'].stdin.write(b'system_reset\n')
                vm['proc'].stdin.flush()
        elif line[0] == 'quit' and vm is not None:
            if vm['proc'] is None:
                print("VM {} not running".format(line[1]))
            else:
                vm['proc'].stdin.write(b'quit\n')
                vm['proc'].stdin.flush()
                try:
                    vm['proc'].wait(60)
                except:
                    pass
                vm['proc'] = None
                print("VM {} has quit".format(line[1]))

KEYPAD_MAP = {'KEY_KP1': 'start 1',
              'KEY_KP4': 'reset 1',
              'KEY_KP7': 'stop 1',
              'KEY_TAB': 'quit 1',}

def handle_keypad(event):
    if event.keystate == event.key_up and event.keycode != 'KEY_NUMLOCK':
        if event.keycode in KEYPAD_MAP:
            handle_stdin(KEYPAD_MAP[event.keycode])
        else:
            print(event, "unhandled")

def handle_inputs ():
    sel = selectors.DefaultSelector()
    try:
        ctrlkp = evdev.InputDevice('/dev/input/by-id/usb-04d9_a02a-event-kbd')
        sel.register(ctrlkp, selectors.EVENT_READ)
    except:
        print("No Keypad found!")
    sel.register(sys.stdin, selectors.EVENT_READ)
    while True:
        for vmname, vm in handlers.items():
            if vm['proc'] is not None and vm['proc'].poll() is not None:
                vm['proc'] = None
                print("VM {} has quit".format(vmname))
        for key, mask in sel.select():
            device = key.fileobj
            if isinstance(device, evdev.InputDevice):
                for event in device.read():
                    event = evdev.util.categorize(event)
                    if isinstance(event, evdev.events.KeyEvent):
                        handle_keypad(event)
            else:
                for l in device.readlines(1):
                    handle_stdin(l)
    

if __name__ == '__main__':
    prepare_configs()
    vms = list(handlers.keys())
    vms.sort()
    print("Found {} VM configs:".format(len(vms)))
    for vm in vms:
        print("\t{}\t{}".format(vm, handlers[vm]['runner']))
    handle_inputs()
