#!/usr/bin/env python3

import json
import os
import sys

PWD = os.path.abspath(os.path.dirname(__file__))
DEFAULTS = {
    'vmid': 0,
    'cpusockets': 1,
    'cpucores': 1,
    'cputhreads': 1,
    'ram': '2G',
    'usb': [],
    'vfio': [],
    'predrives': '',
    'drives': {
        'raw': [],
        'iso': [],
        }
}

def readjson(filename):
    config = json.load(open(filename))
    if not 'vmid' in config:
        # try to infer vmid from filename
        vmid = os.path.splitext(os.path.basename(filename))[0]
        if vmid.isdigit():
            config['vmid'] = int(vmid)
    return config

def assert_unbound_console ():
    if 'efi-framebuffer.0' in os.listdir('/sys/bus/platform/drivers/efi-framebuffer'):
        open('/sys/class/vtconsole/vtcon0/bind', 'w').write('0')
        open('/sys/class/vtconsole/vtcon1/bind', 'w').write('0')
        open('/sys/bus/platform/drivers/efi-framebuffer/unbind', 'w').write('efi-framebuffer.0')

def assert_vfio (addr):
    # we only have the one bus 0000
    addr = '0000:{}'.format(addr)
    if addr not in os.listdir('/sys/bus/pci/drivers/vfio-pci'):
        open('/sys/bus/pci/devices/{}/driver/unbind'.format(addr), 'w').write(addr)
        open('/sys/bus/pci/devices/{}/driver_override'.format(addr), 'w').write('vfio-pci')
        open('/sys/bus/pci/drivers/vfio-pci/bind', 'w').write(addr)

def format(config):
    assert_unbound_console()
    actualconfig = {}
    actualconfig.update(DEFAULTS)
    actualconfig.update(config)
    if not 'vmname' in actualconfig:
        actualconfig['vmname'] = 'vm{}'.format(actualconfig['vmid'])
    if not 'mac' in actualconfig:
        actualconfig['mac'] = '00:16:3e:00:01:{:02x}'.format(actualconfig['vmid'])
    tmp = []
    for e in actualconfig['usb']:
        tmpaddr = []
        if 'hostbus' in e and len(e) == 1:
            for i in range(14):
                tmp.append('\n-device usb-host,hostbus={},hostport={} \\'.format(e['hostbus'], i+1))
        else:
            for k,v in e.items():
                tmpaddr.append('{}={}'.format(k,v))
            tmp.append('\n-device usb-host,{} \\'.format(','.join(tmpaddr)))
    actualconfig['usb'] = ''.join(tmp)
    tmp = []
    for e in actualconfig['vfio']:
        vfioaddr = e.split(',')[0].strip()
        assert_vfio(vfioaddr)
        tmp.append('\n-device vfio-pci,host={} \\'.format(e))
    actualconfig['vfio'] = ''.join(tmp)
    tmp = []
    for e in actualconfig['drives'].get('raw', []):
        tmp.append('\n-drive id=disk{},if=virtio,cache=none,format=raw,file={} \\'.format(
            len(tmp), e))
    for e in actualconfig['drives'].get('iso', []):
        tmp.append('\n  > -drive index={},file={},media=cdrom \\'.format(
            len(tmp), e))
    actualconfig['drives'] = ''.join(tmp)
    actualconfig['cpuunits'] = actualconfig['cpusockets'] * actualconfig['cpucores'] * actualconfig['cputhreads']
    template = open(os.path.join(PWD, 'kvm.template')).read()
    return template.format(**actualconfig)

if __name__ == '__main__':
    if len(sys.argv) > 1:
        config = readjson(sys.argv[1])
    else:
        config = {}
    print(format(config))

