#!/usr/bin/env python3

import json
import os
import sys

PWD = os.path.abspath(os.path.dirname(__file__))
DEFAULTS = {
    'vmid': 0,
    'cpusockets': 1,
    'cpucores': 1,
    'cputhreads': 1,
    'ram': '2G',
    'usb': [],
    'vfio': [],
    'predrives': '',
    'drives': {
        'raw': [],
        'iso': [],
        }
}

def readjson(filename):
    config = json.load(open(filename))
    if not 'vmid' in config:
        # try to infer vmid from filename
        vmid = os.path.splitext(os.path.basename(filename))[0]
        if vmid.isdigit():
            config['vmid'] = int(vmid)
    return config

def format(config):
    actualconfig = {}
    actualconfig.update(DEFAULTS)
    actualconfig.update(config)
    if not 'vmname' in actualconfig:
        actualconfig['vmname'] = 'vm{}'.format(actualconfig['vmid'])
    if not 'mac' in actualconfig:
        actualconfig['mac'] = '00:16:3e:00:01:{:02x}'.format(actualconfig['vmid'])
    tmp = []
    for e in actualconfig['usb']:
        tmp.append('\n-device usb-host,vendorid=0x{:04x},productid=0x{:04x} \\'.format(*e))
    actualconfig['usb'] = ''.join(tmp)
    tmp = []
    for e in actualconfig['vfio']:
        tmp.append('\n-device vfio-pci,host={} \\'.format(e))
    actualconfig['vfio'] = ''.join(tmp)
    tmp = []
    for e in actualconfig['drives'].get('raw', []):
        tmp.append('\n-drive id=disk{},if=virtio,cache=none,format=raw,file={} \\'.format(
            len(tmp), e))
    for e in actualconfig['drives'].get('iso', []):
        tmp.append('\n  > -drive index={},file={},media=cdrom \\'.format(
            len(tmp), e))
    actualconfig['drives'] = ''.join(tmp)
    actualconfig['cpuunits'] = actualconfig['cpusockets'] * actualconfig['cpucores'] * actualconfig['cputhreads']
    template = open(os.path.join(PWD, 'kvm.template')).read()
    return template.format(**actualconfig)

if __name__ == '__main__':
    if len(sys.argv) > 1:
        config = readjson(sys.argv[1])
    else:
        config = {}
    print(format(config))

